#!/usr/bin/env node

/**
 * Remotion TSX Drag-and-Drop Renderer aicontroversy.com
 * Usage: node render-remotion.js <tsx-file-path> [output-dir]
 */

const { bundle } = require('@remotion/bundler');
const { renderMedia, selectComposition, getCompositions } = require('@remotion/renderer');
const path = require('path');
const fs = require('fs');

// Configuration
const CONFIG = {
  defaultOutputDir: './rendered-videos',
  codec: 'h264',
  crf: 18,
  pixelFormat: 'yuv420p'
};

// Check if TSX file contains mock Remotion implementations
function hasMockImplementations(content) {
  return content.includes('REMOTION MOCK') || 
         content.includes('mock of the Remotion API') ||
         (content.includes('const useCurrentFrame = () =>') && content.includes('setInterval'));
}

// Convert mock Remotion code to real Remotion imports
function convertMockToRealRemotion(content) {
  console.log('🔄 Detected Canvas preview code - converting to production Remotion...');
  
  // Remove everything before the PRE-GENERATED DATA section or MAIN COMPONENT
  const preGenMatch = content.match(/\/\/ ---+ PRE-GENERATED DATA ---+/i);
  const mainCompMatch = content.match(/\/\/ ---+ MAIN COMPONENT ---+/i);
  
  let startIndex = 0;
  if (preGenMatch) {
    startIndex = preGenMatch.index;
  } else if (mainCompMatch) {
    startIndex = mainCompMatch.index;
  }
  
  // Keep everything from the data/component section onwards
  let cleanedContent = content.substring(startIndex);
  
  // Add proper Remotion imports at the top
  const properImports = `import React from 'react';
import { useCurrentFrame, useVideoConfig, interpolate, Easing, AbsoluteFill, Sequence, spring } from 'remotion';

`;
  
  // Remove mock implementations and their comments
  cleanedContent = cleanedContent.replace(/\/\*\*[\s\S]*?mock[\s\S]*?\*\//gi, '');
  cleanedContent = cleanedContent.replace(/\/\/ ---+ REMOTION MOCK SYSTEM ---+[\s\S]*?(?=\/\/ ---+ PRE-GENERATED DATA|\/\/ ---+ MAIN COMPONENT)/i, '');
  
  // Combine
  return properImports + cleanedContent;
}

// Ensure the file has proper exports
function ensureProperExports(content, fileName) {
  const componentNameMatch = content.match(/const\s+(\w+):\s*React\.FC/);
  const componentName = componentNameMatch ? componentNameMatch[1] : fileName.replace(/[^a-zA-Z0-9]/g, '');
  
  // Check if default export exists
  const hasDefaultExport = content.includes('export default');
  const hasRemotionVideoExport = content.includes('export const RemotionVideo');
  
  let additions = '';
  
  if (!hasDefaultExport) {
    additions += `\nexport default ${componentName};\n`;
  }
  
  if (!hasRemotionVideoExport) {
    additions += `
export const RemotionVideo = {
  component: ${componentName},
  durationInFrames: 300,
  fps: 30,
  width: 1920,
  height: 1080,
  id: '${componentName}',
};
`;
  }
  
  return content + additions;
}

// Create a temporary entry point that registers the TSX component
function createEntryPoint(tsxFilePath) {
  const tempDir = path.join(__dirname, '.temp-remotion');
  if (!fs.existsSync(tempDir)) {
    fs.mkdirSync(tempDir, { recursive: true });
  }

  const fileName = path.basename(tsxFilePath, path.extname(tsxFilePath));
  const entryPath = path.join(tempDir, 'entry.tsx');
  const absoluteTsxPath = path.resolve(tsxFilePath);
  
  // Read the original file
  let originalContent = fs.readFileSync(absoluteTsxPath, 'utf-8');
  
  // Check if it has mock implementations
  if (hasMockImplementations(originalContent)) {
    originalContent = convertMockToRealRemotion(originalContent);
    originalContent = ensureProperExports(originalContent, fileName);
    
    // Write the cleaned version to a temp file
    const cleanedTsxPath = path.join(tempDir, `${fileName}_cleaned.tsx`);
    fs.writeFileSync(cleanedTsxPath, originalContent);
    console.log('✅ Converted to production Remotion code');
    
    // Use the cleaned file
    return createEntryPointFromCleanedFile(cleanedTsxPath, fileName);
  }

  // Original logic for proper Remotion files
  const entryContent = `
import React from 'react';
import { Composition } from 'remotion';
import { registerRoot } from 'remotion';

// Import the user's component and config
import UserComponent, { RemotionVideo } from '${absoluteTsxPath.replace(/\\/g, '/')}';

// Default configuration
const defaultConfig = {
  component: UserComponent,
  durationInFrames: 300,
  fps: 30,
  width: 1920,
  height: 1080,
  id: '${fileName}'
};

// Merge with user's config if available
const config = RemotionVideo ? {
  ...defaultConfig,
  ...RemotionVideo,
  component: RemotionVideo.component || UserComponent
} : defaultConfig;

const Root: React.FC = () => {
  return (
    <>
      <Composition
        id={config.id}
        component={config.component}
        durationInFrames={config.durationInFrames}
        fps={config.fps}
        width={config.width}
        height={config.height}
        defaultProps={config.defaultProps || {}}
      />
    </>
  );
};

registerRoot(Root);
`;

  fs.writeFileSync(entryPath, entryContent);
  return entryPath;
}

// Create entry point for cleaned/converted files
function createEntryPointFromCleanedFile(cleanedTsxPath, fileName) {
  const tempDir = path.dirname(cleanedTsxPath);
  const entryPath = path.join(tempDir, 'entry.tsx');
  
  const entryContent = `
import React from 'react';
import { Composition } from 'remotion';
import { registerRoot } from 'remotion';

// Import the cleaned component
import UserComponent, { RemotionVideo } from '${cleanedTsxPath.replace(/\\/g, '/')}';

const defaultConfig = {
  component: UserComponent,
  durationInFrames: 300,
  fps: 30,
  width: 1920,
  height: 1080,
  id: '${fileName}'
};

const config = RemotionVideo ? {
  ...defaultConfig,
  ...RemotionVideo,
  component: RemotionVideo.component || UserComponent
} : defaultConfig;

const Root: React.FC = () => {
  return (
    <>
      <Composition
        id={config.id}
        component={config.component}
        durationInFrames={config.durationInFrames}
        fps={config.fps}
        width={config.width}
        height={config.height}
        defaultProps={config.defaultProps || {}}
      />
    </>
  );
};

registerRoot(Root);
`;

  fs.writeFileSync(entryPath, entryContent);
  return entryPath;
}

async function renderTsxFile(tsxFilePath, outputDir = CONFIG.defaultOutputDir) {
  let entryPath = null;
  
  try {
    console.log('🎬 Starting Remotion rendering process...\n');
    
    // Validate input file
    if (!fs.existsSync(tsxFilePath)) {
      throw new Error(`File not found: ${tsxFilePath}`);
    }

    const absoluteTsxPath = path.resolve(tsxFilePath);
    const fileName = path.basename(tsxFilePath, path.extname(tsxFilePath));
    
    console.log(`📄 Input file: ${absoluteTsxPath}`);
    
    // Create output directory if it doesn't exist
    if (!fs.existsSync(outputDir)) {
      fs.mkdirSync(outputDir, { recursive: true });
      console.log(`📁 Created output directory: ${outputDir}`);
    }

    // Create temporary entry point (handles both mock and real Remotion)
    console.log('\n🔧 Processing file...');
    entryPath = createEntryPoint(absoluteTsxPath);
    console.log(`✅ Entry point ready`);

    // Step 1: Bundle the TSX file
    console.log('\n📦 Bundling TSX file...');
    const bundleLocation = await bundle({
      entryPoint: entryPath,
      webpackOverride: (config) => config,
    });
    console.log(`✅ Bundle created at: ${bundleLocation}`);

    // Step 2: Get all compositions
    console.log('\n🔍 Reading compositions...');
    const compositions = await getCompositions(bundleLocation, {
      inputProps: {},
    });
    
    if (compositions.length === 0) {
      throw new Error('No compositions found in the TSX file');
    }

    // Use the first composition (or the one matching the filename)
    let composition = compositions.find(c => c.id === fileName) || compositions[0];
    
    console.log(`✅ Found composition: "${composition.id}"`);
    console.log(`   Duration: ${composition.durationInFrames} frames @ ${composition.fps}fps`);
    console.log(`   Resolution: ${composition.width}x${composition.height}`);

    // Step 3: Render to MP4
    const outputPath = path.join(outputDir, `${fileName}.mp4`);
    console.log(`\n🎥 Rendering video to: ${outputPath}`);
    console.log('⏳ This may take a while...\n');

    await renderMedia({
      composition,
      serveUrl: bundleLocation,
      codec: CONFIG.codec,
      outputLocation: outputPath,
      inputProps: {},
      crf: CONFIG.crf,
      pixelFormat: CONFIG.pixelFormat,
      onProgress: ({ progress, renderedFrames, encodedFrames }) => {
        const percent = (progress * 100).toFixed(1);
        process.stdout.write(
          `\r⏳ Progress: ${percent}% | Rendered: ${renderedFrames}/${composition.durationInFrames} frames | Encoded: ${encodedFrames}`
        );
      },
    });

    console.log('\n\n✅ Rendering complete!');
    console.log(`📹 Video saved to: ${outputPath}`);
    console.log(`📊 File size: ${(fs.statSync(outputPath).size / 1024 / 1024).toFixed(2)} MB`);
    
    return outputPath;

  } catch (error) {
    console.error('\n❌ Error during rendering:');
    console.error(error.message);
    
    if (error.stack) {
      console.error('\nStack trace:');
      console.error(error.stack);
    }
    
    process.exit(1);
  } finally {
    // Cleanup temporary files
    if (entryPath && fs.existsSync(entryPath)) {
      try {
        const tempDir = path.dirname(entryPath);
        fs.rmSync(tempDir, { recursive: true, force: true });
      } catch (e) {
        // Ignore cleanup errors
      }
    }
  }
}

// Main execution
const args = process.argv.slice(2);

if (args.length === 0) {
  console.error('❌ No TSX file provided!');
  console.error('\nUsage: node render-remotion.js <tsx-file-path> [output-dir]');
  console.error('Example: node render-remotion.js ./MyVideo.tsx ./output');
  process.exit(1);
}

const [tsxFile, outputDir] = args;
renderTsxFile(tsxFile, outputDir);